# Hostxolo Production Troubleshooting Guide

## 🔧 Common Issues & Solutions

This guide covers common issues that may occur on live servers and their solutions.

---

## 🚨 Critical Issues

### Issue 1: White Screen / Blank Page
**Symptoms**: Completely blank page, no error messages

**Causes & Solutions**:

#### 1. PHP Fatal Error
```bash
# Check PHP error logs
# Location: /home/username/logs/error_log or cPanel Error Log

# Common fixes:
# 1. Check syntax errors in recently modified files
# 2. Verify all required files exist
# 3. Check file permissions
```

#### 2. Memory Limit Exceeded
```php
// Add to .htaccess or php.ini
php_value memory_limit 256M
php_value max_execution_time 300
```

#### 3. Database Connection Failed
```php
// Test database connection
<?php
try {
    $pdo = new PDO("mysql:host=localhost;dbname=your_db", "user", "pass");
    echo "Database connected successfully";
} catch(PDOException $e) {
    echo "Connection failed: " . $e->getMessage();
}
?>
```

---

### Issue 2: Database Connection Errors
**Symptoms**: "Could not connect to database" or SQL errors

**Solutions**:

#### 1. Verify .env Configuration
```bash
# Check these values in .env
DB_HOST=localhost
DB_DATABASE=correct_db_name
DB_USERNAME=correct_username
DB_PASSWORD=correct_password
DB_CHARSET=utf8mb4
```

#### 2. Check Database User Permissions
```sql
-- In phpMyAdmin, run:
SHOW GRANTS FOR 'hostxolo_user'@'localhost';

-- Should show ALL PRIVILEGES on the database
```

#### 3. Test Database Connection
Create `test-db.php`:
```php
<?php
require_once 'config.php';
try {
    $stmt = db()->query("SELECT 1");
    echo "Database connection: OK";
} catch(Exception $e) {
    echo "Database error: " . $e->getMessage();
}
?>
```

---

### Issue 3: Payment Gateway Failures
**Symptoms**: Payment processing fails, webhook issues

**Solutions**:

#### 1. Razorpay Configuration
```bash
# Check .env settings
RAZORPAY_MODE=live  # Should be 'live' for production
RAZORPAY_LIVE_KEY_ID=rzp_live_xxx
RAZORPAY_LIVE_KEY_SECRET=xxx

# Test API connection
curl -X POST https://api.razorpay.com/v1/payments \
-u key_id:key_secret \
-d "amount=50000&currency=INR&receipt=test#1"
```

#### 2. Webhook Issues
```bash
# Verify webhook URL is accessible
curl -X POST https://yourdomain.com/hostxolo/webhooks/razorpay.php

# Check webhook logs in Razorpay dashboard
```

#### 3. SSL Certificate
```bash
# Ensure SSL is valid
openssl s_client -connect yourdomain.com:443
```

---

## 📧 Email & Notification Issues

### Issue 4: Email Not Sending
**Symptoms**: Users don't receive emails, notifications fail

**Solutions**:

#### 1. SMTP Configuration Test
```php
// Create test-smtp.php
<?php
require_once 'config.php';
require_once 'includes/email-functions.php';

$result = send_email(
    'your-test@email.com',
    'Test User',
    'Test Email',
    '<h1>Test Email</h1><p>This is a test email.</p>'
);

echo $result ? "Email sent successfully" : "Email failed";
?>
```

#### 2. Check SMTP Settings
```bash
# Verify these in .env
SMTP_HOST=smtp.yourdomain.com
SMTP_PORT=587
SMTP_ENCRYPTION=tls
SMTP_USERNAME=noreply@yourdomain.com
SMTP_PASSWORD=correct_password

# Common SMTP ports:
# 25 - No encryption
# 587 - TLS
# 465 - SSL
```

#### 3. Email Authentication
```bash
# Set up SPF, DKIM, and DMARC in cPanel
# SPF: v=spf1 include:spf.yourdomain.com ~all
# DKIM: Generate in cPanel Email Authentication
# DMARC: v=DMARC1; p=quarantine; rua=mailto:dmarc@yourdomain.com
```

---

### Issue 5: SMS/WhatsApp Notifications Not Working
**Symptoms**: No SMS or WhatsApp messages sent

**Solutions**:

#### 1. MSG91 Configuration
```bash
# Check .env settings
SMS_GATEWAY=msg91
SMS_API_KEY=your_api_key
SMS_API_SECRET=your_auth_key
SMS_SENDER_ID=YOURBRAND

# Test API connection
curl -X POST https://api.msg91.com/api/v5/flow/ \
-H "authkey: your_api_key" \
-H "Content-Type: application/json" \
-d '{"template_id":"your_template","recipients":[{"mobiles":"919876543210"}]}'
```

#### 2. WhatsApp Business API
```bash
# Verify credentials
WHATSAPP_API_KEY=your_token
WHATSAPP_PHONE_NUMBER_ID=your_phone_id

# Test webhook
curl -X POST https://graph.facebook.com/v17.0/your_phone_id/messages \
-H "Authorization: Bearer your_token" \
-H "Content-Type: application/json" \
-d '{"messaging_product":"whatsapp","to":"919876543210","type":"text","text":{"body":"Test"}}'
```

---

## 🔒 Security Issues

### Issue 6: Admin Panel Access Issues
**Symptoms**: Can't access admin panel, login failures

**Solutions**:

#### 1. Check Admin Credentials
```sql
-- Reset admin password in phpMyAdmin
UPDATE admin_users 
SET password = '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi' 
WHERE email = 'admin@yourdomain.com';
-- New password: password
```

#### 2. Session Issues
```php
// Check session configuration
phpinfo(); // Look for session settings

// Clear sessions
// In cPanel: PHP Session Manager → Clear Sessions
```

#### 3. File Permissions
```bash
# Correct permissions for cPanel
find /home/username/public_html/hostxolo -type d -exec chmod 755 {} \;
find /home/username/public_html/hostxolo -type f -exec chmod 644 {} \;
chmod 600 /home/username/public_html/hostxolo/.env
```

---

### Issue 7: SSL/HTTPS Issues
**Symptoms**: Mixed content warnings, insecure content

**Solutions**:

#### 1. Force HTTPS
Add to `.htaccess`:
```apache
RewriteEngine On
RewriteCond %{HTTPS} off
RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
```

#### 2. Fix Mixed Content
```bash
# Update hardcoded HTTP URLs in database
UPDATE users SET profile_image = REPLACE(profile_image, 'http://', 'https://');
UPDATE system_settings SET value = REPLACE(value, 'http://', 'https://');
```

#### 3. Check SSL Certificate
```bash
# Test SSL configuration
https://www.ssllabs.com/ssltest/

# Check certificate expiry
openssl s_client -connect yourdomain.com:443 | openssl x509 -noout -dates
```

---

## 🚀 Performance Issues

### Issue 8: Slow Page Loading
**Symptoms**: Website loads slowly, high TTFB

**Solutions**:

#### 1. Enable PHP OPcache
```bash
# In cPanel MultiPHP INI Editor
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=4000
opcache.revalidate_freq=60
```

#### 2. Database Optimization
```sql
-- Add indexes to frequently queried columns
ALTER TABLE user_subscriptions ADD INDEX idx_user_status (user_id, status);
ALTER TABLE support_tickets ADD INDEX idx_status_created (status, created_at);

-- Optimize tables
OPTIMIZE TABLE users, user_subscriptions, support_tickets;
```

#### 3. Enable Caching
```apache
# Add to .htaccess
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType image/png "access plus 1 month"
    ExpiresByType image/jpg "access plus 1 month"
    ExpiresByType image/jpeg "access plus 1 month"
</IfModule>
```

---

## 📱 Mobile & Responsive Issues

### Issue 9: Mobile Display Problems
**Symptoms**: Website not mobile-friendly, touch issues

**Solutions**:

#### 1. Viewport Meta Tag
```html
<!-- Ensure this is in header.php -->
<meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
```

#### 2. Test Mobile Responsiveness
```bash
# Use Chrome DevTools
# 1. Press F12
# 2. Click device icon
# 3. Test different screen sizes
```

#### 3. Touch-Friendly Buttons
```css
/* Ensure buttons are at least 44px */
.btn {
    min-height: 44px;
    min-width: 44px;
}
```

---

## 🔍 Debugging Tools

### Error Logging Setup
```php
// Add to config.php for development
if (APP_ENV === 'local') {
    ini_set('display_errors', 1);
    ini_set('display_startup_errors', 1);
    error_reporting(E_ALL);
} else {
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    error_log('/home/username/public_html/hostxolo/logs/php-errors.log');
}
```

### Database Query Debugging
```php
// Add this to debug database queries
function debug_query($stmt) {
    $stmt->execute();
    echo "<pre>";
    print_r($stmt->errorInfo());
    echo "</pre>";
}
```

### API Response Testing
```bash
# Test payment verification
curl -X POST https://yourdomain.com/hostxolo/payment-verify.php \
-d "subscription_id=1&txn_id=test&razorpay_payment_id=test"

# Test webhook endpoint
curl -X POST https://yourdomain.com/hostxolo/webhooks/razorpay.php \
-H "Content-Type: application/json" \
-d '{"event":"payment.captured"}'
```

---

## 📋 Regular Maintenance Checklist

### Daily
- [ ] Check error logs
- [ ] Monitor disk space
- [ ] Verify email delivery
- [ ] Check payment processing

### Weekly
- [ ] Review security logs
- [ ] Test backup restoration
- [ ] Update software patches
- [ ] Monitor performance metrics

### Monthly
- [ ] Review SSL certificates
- [ ] Update API credentials
- [ ] Clean old logs
- [ ] Security audit

---

## 🚨 Emergency Procedures

### Site Down
1. Check server status with hosting provider
2. Review recent changes
3. Restore from backup if needed
4. Check error logs

### Data Loss
1. Stop all writes to database
2. Restore from latest backup
3. Verify data integrity
4. Communicate with users

### Security Breach
1. Change all passwords
2. Review access logs
3. Update all API keys
4. Enable additional security measures

---

## 📞 Getting Help

### When to Contact Hosting Provider
- Server-level issues
- Network connectivity problems
- Hardware failures
- DNS issues

### When to Contact Developer
- Application-specific bugs
- Feature requests
- Custom modifications
- Performance optimization

### Useful Resources
- [cPanel Documentation](https://docs.cpanel.net/)
- [PHP Documentation](https://www.php.net/docs.php)
- [MySQL Documentation](https://dev.mysql.com/doc/)
- [Razorpay Documentation](https://razorpay.com/docs/)

---

## 🎯 Quick Reference

### Important File Locations
```
/home/username/public_html/hostxolo/
├── .env                    # Environment configuration
├── config.php             # Main configuration
├── includes/              # Core includes
├── logs/                  # Error logs
├── cron/                  # Cron jobs
└── backups/               # Backup files
```

### Common Commands
```bash
# Check PHP version
php -v

# Check MySQL version
mysql --version

# Test database connection
mysql -u username -p database_name

# Check file permissions
ls -la /home/username/public_html/hostxolo/

# View error logs
tail -f /home/username/logs/error_log
```

### Emergency Contacts
- Hosting Provider: [Your Host's Support]
- Domain Registrar: [Your Registrar]
- Payment Gateway: Razorpay Support
- Email Provider: [Your Email Host]

Remember: Always test changes in a staging environment before applying to production!
