<?php
/**
 * Admin - Edit Software Category
 */
require_once __DIR__ . '/../config.php';
require_admin_login();

$page_title = 'Edit Software Category';
$cat_id = intval($_GET['id'] ?? 0);

if (!$cat_id) {
    redirect(BASE_URL . 'admin/software-categories.php', 'error', 'Invalid category ID');
}

// Get category
$category = null;
try {
    $stmt = db()->prepare("SELECT * FROM software_categories WHERE id = ?");
    $stmt->execute([$cat_id]);
    $category = $stmt->fetch();
} catch (PDOException $e) {}

if (!$category) {
    redirect(BASE_URL . 'admin/software-categories.php', 'error', 'Category not found');
}

$errors = [];
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token';
    } else {
        $name = sanitize($_POST['name'] ?? '');
        $slug = sanitize($_POST['slug'] ?? '');
        $description = sanitize($_POST['description'] ?? '');
        $icon = sanitize($_POST['icon'] ?? '');
        $sort_order = intval($_POST['sort_order'] ?? 0);
        $status = isset($_POST['status']) ? 1 : 0;
        
        if (empty($name)) $errors[] = 'Category name is required';
        if (empty($slug)) $errors[] = 'Slug is required';
        
        // Check if slug exists for other category
        if (empty($errors)) {
            try {
                $check = db()->prepare("SELECT id FROM software_categories WHERE slug = ? AND id != ?");
                $check->execute([$slug, $cat_id]);
                if ($check->fetch()) {
                    $errors[] = 'Category slug already exists';
                }
            } catch (PDOException $e) {
                $errors[] = 'Database error';
            }
        }
        
        if (empty($errors)) {
            try {
                $stmt = db()->prepare("UPDATE software_categories SET name = ?, slug = ?, description = ?, icon = ?, sort_order = ?, status = ? WHERE id = ?");
                $stmt->execute([$name, $slug, $description, $icon, $sort_order, $status, $cat_id]);
                
                // Refresh category data
                $stmt = db()->prepare("SELECT * FROM software_categories WHERE id = ?");
                $stmt->execute([$cat_id]);
                $category = $stmt->fetch();
                
                $success = 'Category updated successfully';
            } catch (PDOException $e) {
                $errors[] = 'Failed to update category: ' . $e->getMessage();
            }
        }
    }
}

$csrf_token = generate_csrf_token();
include __DIR__ . '/includes/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2 class="fw-bold mb-0">Edit Software Category</h2>
    <a href="software-categories.php" class="btn btn-outline-secondary">
        <i class="bi bi-arrow-left me-2"></i>Back
    </a>
</div>

<?php if (!empty($errors)): ?>
    <div class="alert alert-danger">
        <?php foreach ($errors as $error): ?><p class="mb-0"><?php echo $error; ?></p><?php endforeach; ?>
    </div>
<?php endif; ?>
<?php if ($success): ?>
    <div class="alert alert-success"><?php echo $success; ?></div>
<?php endif; ?>

<div class="card border-0 shadow-sm">
    <div class="card-body p-4">
        <form method="POST">
            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
            
            <div class="row g-3">
                <div class="col-md-6">
                    <label class="form-label">Category Name <span class="text-danger">*</span></label>
                    <input type="text" name="name" class="form-control" value="<?php echo $category['name']; ?>" required>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Slug <span class="text-danger">*</span></label>
                    <input type="text" name="slug" class="form-control" value="<?php echo $category['slug']; ?>" required>
                </div>
                <div class="col-12">
                    <label class="form-label">Description</label>
                    <textarea name="description" class="form-control" rows="2"><?php echo $category['description']; ?></textarea>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Icon Class <small class="text-muted">(Bootstrap Icons)</small></label>
                    <input type="text" name="icon" class="form-control" value="<?php echo $category['icon']; ?>" placeholder="bi-box">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Sort Order</label>
                    <input type="number" name="sort_order" class="form-control" value="<?php echo $category['sort_order']; ?>">
                </div>
                <div class="col-12">
                    <div class="form-check form-switch">
                        <input type="checkbox" name="status" class="form-check-input" id="status" <?php echo $category['status'] ? 'checked' : ''; ?>>
                        <label class="form-check-label" for="status">Active</label>
                    </div>
                </div>
            </div>
            
            <div class="mt-4">
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-save me-2"></i>Update Category
                </button>
                <a href="software-categories.php" class="btn btn-outline-secondary ms-2">Cancel</a>
            </div>
        </form>
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
