<?php
/**
 * Admin - Edit Testimonial
 */
require_once __DIR__ . '/../config.php';
require_admin_login();

$testimonial_id = intval($_GET['id'] ?? 0);

if (!$testimonial_id) {
    redirect(BASE_URL . 'admin/testimonials.php', 'error', 'Invalid testimonial ID');
}

// Get testimonial data
$testimonial = null;
try {
    $stmt = db()->prepare("SELECT * FROM testimonials WHERE id = ?");
    $stmt->execute([$testimonial_id]);
    $testimonial = $stmt->fetch();
} catch (PDOException $e) {}

if (!$testimonial) {
    redirect(BASE_URL . 'admin/testimonials.php', 'error', 'Testimonial not found');
}

$errors = [];
$success = '';

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token';
    } else {
        $name = sanitize($_POST['name'] ?? '');
        $designation = sanitize($_POST['designation'] ?? '');
        $content = sanitize($_POST['content'] ?? '');
        $rating = intval($_POST['rating'] ?? 5);
        $status = sanitize($_POST['status'] ?? '1');
        
        // Validation
        if (empty($name)) {
            $errors[] = 'Name is required';
        }
        if (empty($content)) {
            $errors[] = 'Content is required';
        }
        if ($rating < 1 || $rating > 5) {
            $errors[] = 'Rating must be between 1 and 5';
        }
        
        if (empty($errors)) {
            try {
                $stmt = db()->prepare("
                    UPDATE testimonials 
                    SET name = ?, designation = ?, content = ?, rating = ?, status = ?, updated_at = NOW() 
                    WHERE id = ?
                ");
                $stmt->execute([$name, $designation, $content, $rating, $status, $testimonial_id]);
                
                // Log activity
                log_activity('testimonial_updated', "Updated testimonial: $name", $_SESSION['admin_id'], 'admin');
                
                redirect(BASE_URL . 'admin/testimonials.php', 'success', 'Testimonial updated successfully');
                
            } catch (PDOException $e) {
                $errors[] = 'Failed to update testimonial';
            }
        }
    }
}

$page_title = 'Edit Testimonial';
include __DIR__ . '/includes/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2 class="fw-bold mb-0">Edit Testimonial</h2>
    <a href="testimonials.php" class="btn btn-outline-secondary">
        <i class="bi bi-arrow-left me-2"></i>Back to Testimonials
    </a>
</div>

<?php if (!empty($errors)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <?php foreach ($errors as $error): ?>
            <p class="mb-0"><?php echo $error; ?></p>
        <?php endforeach; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<div class="card border-0 shadow-sm">
    <div class="card-body p-4">
        <form method="POST">
            <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
            
            <div class="row g-3">
                <div class="col-md-6">
                    <label class="form-label">Name *</label>
                    <input type="text" name="name" class="form-control" value="<?php echo $testimonial['name']; ?>" required>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Designation</label>
                    <input type="text" name="designation" class="form-control" value="<?php echo $testimonial['designation'] ?? ''; ?>">
                </div>
                <div class="col-12">
                    <label class="form-label">Content *</label>
                    <textarea name="content" class="form-control" rows="4" required><?php echo $testimonial['content']; ?></textarea>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Rating *</label>
                    <select name="rating" class="form-select" required>
                        <option value="1" <?php echo $testimonial['rating'] == 1 ? 'selected' : ''; ?>>1 Star</option>
                        <option value="2" <?php echo $testimonial['rating'] == 2 ? 'selected' : ''; ?>>2 Stars</option>
                        <option value="3" <?php echo $testimonial['rating'] == 3 ? 'selected' : ''; ?>>3 Stars</option>
                        <option value="4" <?php echo $testimonial['rating'] == 4 ? 'selected' : ''; ?>>4 Stars</option>
                        <option value="5" <?php echo $testimonial['rating'] == 5 ? 'selected' : ''; ?>>5 Stars</option>
                    </select>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Status</label>
                    <select name="status" class="form-select">
                        <option value="1" <?php echo $testimonial['status'] == 1 ? 'selected' : ''; ?>>Active</option>
                        <option value="0" <?php echo $testimonial['status'] == 0 ? 'selected' : ''; ?>>Inactive</option>
                    </select>
                </div>
            </div>
            
            <div class="mt-4">
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-save me-2"></i>Update Testimonial
                </button>
                <a href="testimonials.php" class="btn btn-outline-secondary">Cancel</a>
            </div>
        </form>
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
