<?php
/**
 * Hostxolo - Single Blog Post Page
 */
require_once 'config.php';

// Get the blog post slug from URL
$slug = $_GET['slug'] ?? '';

if (empty($slug)) {
    redirect(BASE_URL . 'blog.php');
}

// Fetch the blog post with category info
try {
    $stmt = db()->prepare("
        SELECT bp.*, bc.name as category_name, bc.slug as category_slug 
        FROM blog_posts bp 
        LEFT JOIN blog_categories bc ON bp.category_id = bc.id 
        WHERE bp.slug = ? AND bp.status = 'published'
    ");
    $stmt->execute([$slug]);
    $post = $stmt->fetch();
} catch (PDOException $e) {
    $post = null;
}

// If post not found, show 404
if (!$post) {
    header("HTTP/1.0 404 Not Found");
    include '404.php';
    exit;
}

// Increment view count
try {
    $stmt = db()->prepare("UPDATE blog_posts SET views = views + 1 WHERE id = ?");
    $stmt->execute([$post['id']]);
} catch (PDOException $e) {
    // Silently fail - not critical
}

// Get related posts (same category, excluding current post)
try {
    $stmt = db()->prepare("
        SELECT bp.*, bc.name as category_name, bc.slug as category_slug 
        FROM blog_posts bp 
        LEFT JOIN blog_categories bc ON bp.category_id = bc.id 
        WHERE bp.status = 'published' AND bp.id != ? 
        AND (bp.category_id = ? OR bp.category_id IS NULL)
        ORDER BY bp.published_at DESC 
        LIMIT 3
    ");
    $stmt->execute([$post['id'], $post['category_id'] ?? null]);
    $related_posts = $stmt->fetchAll();
} catch (PDOException $e) {
    $related_posts = [];
}

// SEO meta data
$seo_title = $post['meta_title'] ?: $post['title'] . ' - Hostxolo Blog';
$seo_description = $post['meta_description'] ?: substr(strip_tags($post['excerpt'] ?? $post['content']), 0, 160);
$seo_keywords = $post['meta_keywords'] ?: '';

// Override SEO variables for header
$GLOBALS['page_seo_title'] = $seo_title;
$GLOBALS['page_seo_description'] = $seo_description;
$GLOBALS['page_seo_keywords'] = $seo_keywords;
$GLOBALS['page_og_image'] = $post['featured_image'] ? UPLOADS_URL . $post['featured_image'] : '';

include 'includes/header.php';
?>

<!-- Page Header -->
<section class="page-header">
    <div class="container">
        <h1><?php echo $post['title']; ?></h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb justify-content-center">
                <li class="breadcrumb-item"><a href="<?php echo BASE_URL; ?>">Home</a></li>
                <li class="breadcrumb-item"><a href="<?php echo BASE_URL; ?>blog.php">Blog</a></li>
                <li class="breadcrumb-item active" aria-current="page"><?php echo substr($post['title'], 0, 30) . (strlen($post['title']) > 30 ? '...' : ''); ?></li>
            </ol>
        </nav>
    </div>
</section>

<!-- Blog Post Content -->
<section class="py-5">
    <div class="container">
        <div class="row">
            <!-- Main Content -->
            <div class="col-lg-8">
                <article class="blog-single">
                    <!-- Post Meta -->
                    <div class="blog-meta d-flex flex-wrap gap-3 mb-3">
                        <?php if ($post['category_name']): ?>
                        <span class="badge bg-primary bg-opacity-10 text-primary">
                            <i class="bi bi-folder me-1"></i>
                            <a href="<?php echo BASE_URL; ?>blog.php?category=<?php echo $post['category_slug']; ?>" class="text-decoration-none">
                                <?php echo $post['category_name']; ?>
                            </a>
                        </span>
                        <?php endif; ?>
                        <span class="text-muted">
                            <i class="bi bi-calendar me-1"></i>
                            <?php echo format_date($post['published_at']); ?>
                        </span>
                        <span class="text-muted">
                            <i class="bi bi-person me-1"></i>
                            <?php echo $post['author_name']; ?>
                        </span>
                        <span class="text-muted">
                            <i class="bi bi-eye me-1"></i>
                            <?php echo number_format($post['views'] + 1); ?> views
                        </span>
                    </div>
                    
                    <!-- Post Title -->
                    <h1 class="blog-title mb-4"><?php echo $post['title']; ?></h1>
                    
                    <!-- Post Excerpt -->
                    <?php if ($post['excerpt']): ?>
                    <div class="lead text-muted mb-4">
                        <?php echo $post['excerpt']; ?>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Post Content -->
                    <div class="blog-content">
                        <?php echo $post['content']; ?>
                    </div>
                    
                    <!-- Share Buttons -->
                    <div class="blog-share mt-5 pt-4 border-top">
                        <h5>Share this article:</h5>
                        <div class="d-flex gap-2">
                            <a href="https://www.facebook.com/sharer/sharer.php?u=<?php echo urlencode(BASE_URL . 'blog-post.php?slug=' . $post['slug']); ?>" 
                               target="_blank" class="btn btn-outline-primary btn-sm">
                                <i class="bi bi-facebook"></i>
                            </a>
                            <a href="https://twitter.com/intent/tweet?url=<?php echo urlencode(BASE_URL . 'blog-post.php?slug=' . $post['slug']); ?>&text=<?php echo urlencode($post['title']); ?>" 
                               target="_blank" class="btn btn-outline-info btn-sm">
                                <i class="bi bi-twitter"></i>
                            </a>
                            <a href="https://www.linkedin.com/shareArticle?mini=true&url=<?php echo urlencode(BASE_URL . 'blog-post.php?slug=' . $post['slug']); ?>&title=<?php echo urlencode($post['title']); ?>" 
                               target="_blank" class="btn btn-outline-primary btn-sm">
                                <i class="bi bi-linkedin"></i>
                            </a>
                            <a href="https://wa.me/?text=<?php echo urlencode($post['title'] . ' - ' . BASE_URL . 'blog-post.php?slug=' . $post['slug']); ?>" 
                               target="_blank" class="btn btn-outline-success btn-sm">
                                <i class="bi bi-whatsapp"></i>
                            </a>
                        </div>
                    </div>
                </article>
            </div>
            
            <!-- Sidebar -->
            <div class="col-lg-4">
                <aside class="blog-sidebar">
                    <!-- Back to Blog -->
                    <div class="card mb-4 border-0 shadow-sm">
                        <div class="card-body text-center">
                            <a href="<?php echo BASE_URL; ?>blog.php" class="btn btn-primary">
                                <i class="bi bi-arrow-left me-2"></i>Back to Blog
                            </a>
                        </div>
                    </div>
                    
                    <!-- Post Info -->
                    <div class="card mb-4 border-0 shadow-sm">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0"><i class="bi bi-info-circle me-2"></i>Article Info</h5>
                        </div>
                        <ul class="list-group list-group-flush">
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Published:</span>
                                <span class="text-muted"><?php echo format_date($post['published_at']); ?></span>
                            </li>
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Author:</span>
                                <span class="text-muted"><?php echo $post['author_name']; ?></span>
                            </li>
                            <?php if ($post['category_name']): ?>
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Category:</span>
                                <a href="<?php echo BASE_URL; ?>blog.php?category=<?php echo $post['category_slug']; ?>" class="text-decoration-none">
                                    <?php echo $post['category_name']; ?>
                                </a>
                            </li>
                            <?php endif; ?>
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Views:</span>
                                <span class="text-muted"><?php echo number_format($post['views'] + 1); ?></span>
                            </li>
                        </ul>
                    </div>
                    
                    <!-- Related Posts -->
                    <?php if (!empty($related_posts)): ?>
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-light">
                            <h5 class="mb-0"><i class="bi bi-journal-text me-2"></i>Related Articles</h5>
                        </div>
                        <div class="list-group list-group-flush">
                            <?php foreach ($related_posts as $related): ?>
                            <a href="<?php echo BASE_URL; ?>blog-post.php?slug=<?php echo $related['slug']; ?>" 
                               class="list-group-item list-group-item-action">
                                <div class="d-flex w-100 justify-content-between">
                                    <h6 class="mb-1"><?php echo substr($related['title'], 0, 50) . (strlen($related['title']) > 50 ? '...' : ''); ?></h6>
                                </div>
                                <small class="text-muted">
                                    <i class="bi bi-calendar me-1"></i><?php echo format_date($related['published_at']); ?>
                                </small>
                            </a>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </aside>
            </div>
        </div>
    </div>
</section>

<!-- Related Posts Section (Full Width) -->
<?php if (!empty($related_posts)): ?>
<section class="py-5 bg-light">
    <div class="container">
        <h3 class="mb-4">You May Also Like</h3>
        <div class="row g-4">
            <?php foreach ($related_posts as $related): ?>
            <div class="col-md-4">
                <article class="card h-100 blog-card border-0 shadow-sm">
                    <div class="blog-image bg-primary bg-opacity-10 d-flex align-items-center justify-content-center" style="height: 180px;">
                        <i class="bi bi-newspaper text-primary" style="font-size: 3rem;"></i>
                    </div>
                    <div class="card-body p-4">
                        <div class="d-flex justify-content-between align-items-center mb-2">
                            <?php if ($related['category_name']): ?>
                                <span class="badge bg-primary bg-opacity-10 text-primary">
                                    <?php echo $related['category_name']; ?>
                                </span>
                            <?php endif; ?>
                            <small class="text-muted">
                                <i class="bi bi-calendar me-1"></i><?php echo format_date($related['published_at']); ?>
                            </small>
                        </div>
                        <h5 class="card-title">
                            <a href="<?php echo BASE_URL; ?>blog-post.php?slug=<?php echo $related['slug']; ?>" class="text-decoration-none">
                                <?php echo substr($related['title'], 0, 60) . (strlen($related['title']) > 60 ? '...' : ''); ?>
                            </a>
                        </h5>
                        <p class="card-text text-muted">
                            <?php echo substr(strip_tags($related['excerpt'] ?? $related['content']), 0, 100) . '...'; ?>
                        </p>
                        <a href="<?php echo BASE_URL; ?>blog-post.php?slug=<?php echo $related['slug']; ?>" class="btn btn-outline-primary btn-sm">
                            Read More <i class="bi bi-arrow-right"></i>
                        </a>
                    </div>
                </article>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<?php include 'includes/footer.php'; ?>
