<?php
/**
 * Hostxolo - Configuration File
 * 
 * This file contains all the configuration settings for the Hostxolo website.
 * Settings are loaded from .env file for easy configuration.
 */

// Include PHPMailer autoloader
if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require_once __DIR__ . '/vendor/autoload.php';
}

// Include email functions
require_once __DIR__ . '/includes/email-functions.php';

// ============================================
// LOAD ENVIRONMENT VARIABLES FROM .ENV FILE
// ============================================
function load_env_file($path) {
    if (!file_exists($path)) {
        return false;
    }
    
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '#') === 0 || strpos($line, '=') === false) {
            continue;
        }
        
        list($key, $value) = explode('=', $line, 2);
        $key = trim($key);
        $value = trim($value);
        
        if (!empty($key)) {
            $_ENV[$key] = $value;
            putenv("$key=$value");
        }
    }
    return true;
}

// Load .env file
load_env_file(__DIR__ . '/.env');

// Helper function to get env variable with default
function env($key, $default = '') {
    $value = $_ENV[$key] ?? getenv($key);
    return $value !== false ? $value : $default;
}

// ============================================
// DATABASE CONFIGURATION
// ============================================
define('DB_HOST', env('DB_HOST', 'localhost'));
define('DB_USERNAME', env('DB_USERNAME', 'root'));
define('DB_PASSWORD', env('DB_PASSWORD', ''));
define('DB_NAME', env('DB_DATABASE', 'hostxolo'));
define('DB_CHARSET', env('DB_CHARSET', 'utf8mb4'));

// ============================================
// SITE CONFIGURATION
// ============================================
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'] ?? 'localhost';
define('BASE_URL', $protocol . '://' . $host . '/hostxolo/');
define('SITE_NAME', 'Hostxolo');
define('SITE_TAGLINE', 'Your Business Growth Partner');

// ============================================
// PATH CONFIGURATION
// ============================================
define('ROOT_PATH', __DIR__ . '/');
define('ASSETS_PATH', ROOT_PATH . 'assets/');
define('UPLOADS_PATH', ROOT_PATH . 'uploads/');
define('INCLUDES_PATH', ROOT_PATH . 'includes/');
define('ADMIN_PATH', ROOT_PATH . 'admin/');
define('USER_PATH', ROOT_PATH . 'user/');

// URL Paths
define('ASSETS_URL', BASE_URL . 'assets/');
define('UPLOADS_URL', BASE_URL . 'uploads/');

// ============================================
// SECURITY CONFIGURATION
// ============================================
define('SESSION_NAME', 'hostxolo_session');
define('SESSION_LIFETIME', 7200); // 2 hours
define('CSRF_TOKEN_NAME', 'csrf_token');
define('MAX_LOGIN_ATTEMPTS', 5);
define('LOGIN_LOCKOUT_TIME', 900); // 15 minutes

// Encryption Keys (Change these for production!)
define('ENCRYPTION_KEY', env('ENCRYPTION_KEY', 'your-secret-encryption-key-change-this'));
define('JWT_SECRET', env('JWT_SECRET', 'your-jwt-secret-key-change-this'));

// ============================================
// PAYMENT GATEWAY CONFIGURATION
// ============================================

// Razorpay
$razorpay_mode = env('RAZORPAY_MODE', 'test');
define('RAZORPAY_MODE', $razorpay_mode);
if ($razorpay_mode === 'live') {
    define('RAZORPAY_KEY_ID', env('RAZORPAY_LIVE_KEY_ID', ''));
    define('RAZORPAY_KEY_SECRET', env('RAZORPAY_LIVE_KEY_SECRET', ''));
    define('RAZORPAY_WEBHOOK_SECRET', env('RAZORPAY_LIVE_WEBHOOK_SECRET', ''));
} else {
    define('RAZORPAY_KEY_ID', env('RAZORPAY_TEST_KEY_ID', ''));
    define('RAZORPAY_KEY_SECRET', env('RAZORPAY_TEST_KEY_SECRET', ''));
    define('RAZORPAY_WEBHOOK_SECRET', env('RAZORPAY_TEST_WEBHOOK_SECRET', ''));
}

// Stripe
$stripe_mode = env('STRIPE_MODE', 'test');
if ($stripe_mode === 'live') {
    define('STRIPE_PUBLISHABLE_KEY', env('STRIPE_LIVE_PUBLISHABLE_KEY', ''));
    define('STRIPE_SECRET_KEY', env('STRIPE_LIVE_SECRET_KEY', ''));
    define('STRIPE_WEBHOOK_SECRET', env('STRIPE_LIVE_WEBHOOK_SECRET', ''));
} else {
    define('STRIPE_PUBLISHABLE_KEY', env('STRIPE_TEST_PUBLISHABLE_KEY', ''));
    define('STRIPE_SECRET_KEY', env('STRIPE_TEST_SECRET_KEY', ''));
    define('STRIPE_WEBHOOK_SECRET', env('STRIPE_TEST_WEBHOOK_SECRET', ''));
}

// PayPal
$paypal_mode = env('PAYPAL_MODE', 'sandbox');
if ($paypal_mode === 'live') {
    define('PAYPAL_CLIENT_ID', env('PAYPAL_LIVE_CLIENT_ID', ''));
    define('PAYPAL_CLIENT_SECRET', env('PAYPAL_LIVE_CLIENT_SECRET', ''));
} else {
    define('PAYPAL_CLIENT_ID', env('PAYPAL_SANDBOX_CLIENT_ID', ''));
    define('PAYPAL_CLIENT_SECRET', env('PAYPAL_SANDBOX_CLIENT_SECRET', ''));
}
define('PAYPAL_SANDBOX', $paypal_mode !== 'live');

// Currency
define('DEFAULT_CURRENCY', env('DEFAULT_CURRENCY', 'INR'));
define('CURRENCY_SYMBOL', '₹');

// ============================================
// EMAIL/SMTP CONFIGURATION
// ============================================
define('SMTP_HOST', env('SMTP_HOST', 'smtp.gmail.com'));
define('SMTP_PORT', env('SMTP_PORT', 587));
define('SMTP_USERNAME', env('SMTP_USERNAME', ''));
define('SMTP_PASSWORD', env('SMTP_PASSWORD', ''));
define('SMTP_ENCRYPTION', env('SMTP_ENCRYPTION', 'tls'));
define('SMTP_FROM_EMAIL', env('SMTP_FROM_EMAIL', 'noreply@hostxolo.com'));
define('SMTP_FROM_NAME', env('SMTP_FROM_NAME', 'Hostxolo'));

// ============================================
// SMS API CONFIGURATION
// ============================================
define('SMS_GATEWAY', env('SMS_GATEWAY', '')); // twilio, msg91, textlocal
define('SMS_API_KEY', env('SMS_API_KEY', ''));
define('SMS_API_SECRET', env('SMS_API_SECRET', ''));
define('SMS_SENDER_ID', env('SMS_SENDER_ID', 'HOSTXO'));

// ============================================
// WHATSAPP API CONFIGURATION
// ============================================
define('WHATSAPP_API_KEY', env('WHATSAPP_API_KEY', ''));
define('WHATSAPP_API_URL', env('WHATSAPP_API_URL', ''));
define('WHATSAPP_BUSINESS_ID', env('WHATSAPP_BUSINESS_ID', ''));

// ============================================
// FILE UPLOAD CONFIGURATION
// ============================================
define('MAX_UPLOAD_SIZE', 5242880); // 5MB
define('MAX_FILE_UPLOAD_SIZE_MB', env('MAX_FILE_UPLOAD_SIZE_MB', 5));
define('ALLOWED_IMAGE_TYPES', ['image/jpeg', 'image/png', 'image/gif', 'image/webp']);
define('ALLOWED_DOCUMENT_TYPES', ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document']);

// ============================================
// GOOGLE SERVICES CONFIGURATION
// ============================================
define('GOOGLE_ANALYTICS_ID', env('GOOGLE_ANALYTICS_ID', ''));
define('GOOGLE_TAG_MANAGER_ID', env('GOOGLE_TAG_MANAGER_ID', ''));
define('GOOGLE_CLIENT_ID', env('GOOGLE_CLIENT_ID', ''));
define('GOOGLE_CLIENT_SECRET', env('GOOGLE_CLIENT_SECRET', ''));
define('GOOGLE_REDIRECT_URI', BASE_URL . 'auth/google-callback.php');

// ============================================
// FACEBOOK LOGIN CONFIGURATION
// ============================================
define('FACEBOOK_APP_ID', env('FACEBOOK_APP_ID', ''));
define('FACEBOOK_APP_SECRET', env('FACEBOOK_APP_SECRET', ''));
define('FACEBOOK_REDIRECT_URI', BASE_URL . 'auth/facebook-callback.php');

// ============================================
// RECAPTCHA CONFIGURATION
// ============================================
define('RECAPTCHA_SITE_KEY', env('RECAPTCHA_SITE_KEY', ''));
define('RECAPTCHA_SECRET_KEY', env('RECAPTCHA_SECRET_KEY', ''));
define('RECAPTCHA_ENABLED', env('RECAPTCHA_ENABLED', 'false') === 'true');

// ============================================
// FEATURE TOGGLES
// ============================================
define('MAINTENANCE_MODE', env('MAINTENANCE_MODE', 'false') === 'true');
define('REGISTRATION_ENABLED', env('REGISTRATION_ENABLED', 'true') === 'true');
define('SOCIAL_LOGIN_ENABLED', env('SOCIAL_LOGIN_ENABLED', 'false') === 'true');
define('EMAIL_VERIFICATION_REQUIRED', env('EMAIL_VERIFICATION_REQUIRED', 'false') === 'true');
define('OTP_ENABLED', env('OTP_ENABLED', 'false') === 'true');
define('REFERRAL_SYSTEM_ENABLED', env('REFERRAL_SYSTEM_ENABLED', 'false') === 'true');
define('COUPON_SYSTEM_ENABLED', env('COUPON_SYSTEM_ENABLED', 'false') === 'true');
define('NEWSLETTER_ENABLED', env('NEWSLETTER_ENABLED', 'true') === 'true');

// ============================================
// SECURITY SETTINGS
// ============================================
define('LOCKOUT_DURATION_MINUTES', env('LOCKOUT_DURATION_MINUTES', 30));
define('SESSION_TIMEOUT_MINUTES', env('SESSION_TIMEOUT_MINUTES', 120));
define('PASSWORD_MIN_LENGTH', env('PASSWORD_MIN_LENGTH', 8));
define('REQUIRE_STRONG_PASSWORD', env('REQUIRE_STRONG_PASSWORD', 'true') === 'true');
define('OTP_EXPIRY_MINUTES', env('OTP_EXPIRY_MINUTES', 10));

// ============================================
// COMPANY INFORMATION
// ============================================
define('COMPANY_NAME', env('COMPANY_NAME', 'Hostxolo Business Solutions Pvt. Ltd.'));
define('COMPANY_GST_NUMBER', env('COMPANY_GST_NUMBER', ''));
define('SUPPORT_PHONE', env('SUPPORT_PHONE', '+91-6388758325'));
define('SUPPORT_EMAIL', env('SUPPORT_EMAIL', 'support@hostxolo.com'));
define('BUSINESS_HOURS', env('BUSINESS_HOURS', 'Mon-Sat: 9:00 AM - 6:00 PM IST'));
define('WHATSAPP_NUMBER', env('WHATSAPP_NUMBER', '+91-6388758325'));

// ============================================
// EMAIL QUEUE SETTINGS
// ============================================
define('EMAIL_QUEUE_ENABLED', true);
define('EMAIL_BATCH_SIZE', 50);
define('EMAIL_RETRY_ATTEMPTS', 3);

// ============================================
// VOICE/TELECALLING API
// ============================================
define('VOICE_ENABLED', false);
define('VOICE_API_KEY', '');
define('VOICE_API_URL', '');

// ============================================
// PAGINATION & LIMITS
// ====================================================
define('ITEMS_PER_PAGE', env('ITEMS_PER_PAGE', 10));
define('BLOG_POSTS_PER_PAGE', 6);
define('MAX_FEATURED_SERVICES', 6);

// ============================================
// ERROR REPORTING
// ============================================
// Set to false in production
define('DEBUG_MODE', true);

if (DEBUG_MODE) {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    ini_set('log_errors', 1);
    ini_set('error_log', ROOT_PATH . 'logs/error.log');
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    ini_set('error_log', ROOT_PATH . 'logs/error.log');
}

// ============================================
// TIMEZONE CONFIGURATION
// ============================================
date_default_timezone_set('Asia/Kolkata');

// ============================================
// DATABASE CONNECTION CLASS
// ============================================
class Database {
    private static ?PDO $instance = null;
    
    public static function getInstance(): PDO {
        if (self::$instance === null) {
            try {
                $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
                self::$instance = new PDO($dsn, DB_USERNAME, DB_PASSWORD, [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false,
                ]);
            } catch (PDOException $e) {
                error_log("Database connection failed: " . $e->getMessage());
                die("Database connection failed. Please try again later.");
            }
        }
        return self::$instance;
    }
    
    private function __clone() {}
    public function __wakeup() {}
}

// ============================================
// HELPER FUNCTIONS
// ============================================

/**
 * Get database connection
 */
function db(): PDO {
    return Database::getInstance();
}

/**
 * Sanitize input data
 */
function sanitize(string $data): string {
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

/**
 * Generate CSRF token
 */
function generate_csrf_token(): string {
    if (empty($_SESSION[CSRF_TOKEN_NAME])) {
        $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
    }
    return $_SESSION[CSRF_TOKEN_NAME];
}

/**
 * Verify CSRF token
 */
function verify_csrf_token(string $token): bool {
    return isset($_SESSION[CSRF_TOKEN_NAME]) && hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
}

/**
 * Generate random string
 */
function generate_random_string(int $length = 10): string {
    return bin2hex(random_bytes($length / 2));
}

/**
 * Format currency
 */
function format_currency(float $amount, string $currency = DEFAULT_CURRENCY): string {
    return '&#8377;' . number_format($amount, 2);
}

/**
 * Format date
 */
function format_date(string $date, string $format = 'd M Y'): string {
    return date($format, strtotime($date));
}

/**
 * Create SEO-friendly slug
 */
function create_slug(string $text): string {
    $text = preg_replace('~[^\pL\d]+~u', '-', $text);
    $text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);
    $text = preg_replace('~[^-\w]+~', '', $text);
    $text = trim($text, '-');
    $text = preg_replace('~-+~', '-', $text);
    $text = strtolower($text);
    return empty($text) ? 'n-a' : $text;
}

/**
 * Redirect with message
 */
function redirect(string $url, string $type = '', string $message = ''): void {
    if (!empty($message)) {
        $_SESSION['flash_' . $type] = $message;
    }
    header("Location: " . $url);
    exit();
}

/**
 * Display flash message
 */
function flash_message(): string {
    $types = ['success', 'error', 'warning', 'info'];
    $html = '';
    
    foreach ($types as $type) {
        if (isset($_SESSION['flash_' . $type])) {
            $html .= '<div class="alert alert-' . ($type === 'error' ? 'danger' : $type) . ' alert-dismissible fade show" role="alert">';
            $html .= $_SESSION['flash_' . $type];
            $html .= '<button type="button" class="btn-close" data-bs-dismiss="alert"></button>';
            $html .= '</div>';
            unset($_SESSION['flash_' . $type]);
        }
    }
    
    return $html;
}

/**
 * Check if user is logged in
 */
function is_logged_in(): bool {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Check if admin is logged in
 */
function is_admin_logged_in(): bool {
    return isset($_SESSION['admin_id']) && !empty($_SESSION['admin_id']);
}

/**
 * Require login
 */
function require_login(): void {
    if (!is_logged_in()) {
        redirect(BASE_URL . 'login.php', 'warning', 'Please login to continue.');
    }
}

/**
 * Require admin login
 */
function require_admin_login(): void {
    if (!is_admin_logged_in()) {
        redirect(BASE_URL . 'admin/login.php', 'warning', 'Please login to continue.');
    }
}

/**
 * Log activity
 */
function log_activity(string $action, string $description = '', int $user_id = null, string $user_type = 'user'): void {
    try {
        $stmt = db()->prepare("INSERT INTO activity_logs (user_id, user_type, action, description, ip_address, user_agent) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $user_id,
            $user_type,
            $action,
            $description,
            $_SERVER['REMOTE_ADDR'] ?? null,
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
    } catch (PDOException $e) {
        error_log("Activity logging failed: " . $e->getMessage());
    }
}

/**
 * Get site setting
 */
function get_setting(string $key, string $default = ''): string {
    try {
        $stmt = db()->prepare("SELECT setting_value FROM site_settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        $result = $stmt->fetch();
        return $result ? $result['setting_value'] : $default;
    } catch (PDOException $e) {
        return $default;
    }
}

/**
 * Upload file
 */
function upload_file(array $file, string $directory, array $allowed_types = ALLOWED_IMAGE_TYPES): array {
    $result = ['success' => false, 'message' => '', 'path' => ''];
    
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $result['message'] = 'Upload failed with error code: ' . $file['error'];
        return $result;
    }
    
    if ($file['size'] > MAX_UPLOAD_SIZE) {
        $result['message'] = 'File size exceeds maximum allowed size.';
        return $result;
    }
    
    if (!in_array($file['type'], $allowed_types)) {
        $result['message'] = 'File type not allowed.';
        return $result;
    }
    
    $upload_dir = UPLOADS_PATH . $directory . '/';
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    $filename = generate_random_string(16) . '_' . time() . '.' . pathinfo($file['name'], PATHINFO_EXTENSION);
    $filepath = $upload_dir . $filename;
    
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        $result['success'] = true;
        $result['path'] = $directory . '/' . $filename;
    } else {
        $result['message'] = 'Failed to move uploaded file.';
    }
    
    return $result;
}

// ============================================
// START SESSION
// ============================================
if (session_status() === PHP_SESSION_NONE) {
    session_name(SESSION_NAME);
    session_start();
}

// Regenerate session ID periodically for security
if (!isset($_SESSION['created'])) {
    $_SESSION['created'] = time();
} else if (time() - $_SESSION['created'] > 1800) {
    session_regenerate_id(true);
    $_SESSION['created'] = time();
}

// ============================================
// SECURITY FUNCTIONS
// ============================================

/**
 * Check if account is locked due to failed login attempts
 */
function is_account_locked(string $username, string $ip_address, string $type = 'user'): bool {
    try {
        $stmt = db()->prepare("SELECT COUNT(*) as attempts FROM login_attempts 
            WHERE (username = ? OR ip_address = ?) 
            AND attempt_type = ? 
            AND status = 'failed' 
            AND created_at > DATE_SUB(NOW(), INTERVAL ? MINUTE)");
        $stmt->execute([$username, $ip_address, $type, LOCKOUT_DURATION_MINUTES]);
        $result = $stmt->fetch();
        return $result['attempts'] >= MAX_LOGIN_ATTEMPTS;
    } catch (PDOException $e) {
        return false;
    }
}

/**
 * Record login attempt
 */
function record_login_attempt(string $username, string $ip_address, string $status, string $type = 'user', string $reason = ''): void {
    try {
        $stmt = db()->prepare("INSERT INTO login_attempts 
            (username, ip_address, user_agent, attempt_type, status, failure_reason) 
            VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $username,
            $ip_address,
            $_SERVER['HTTP_USER_AGENT'] ?? '',
            $type,
            $status,
            $reason
        ]);
    } catch (PDOException $e) {
        error_log("Failed to record login attempt: " . $e->getMessage());
    }
}

/**
 * Check password strength
 */
function check_password_strength(string $password): array {
    $errors = [];
    
    if (strlen($password) < PASSWORD_MIN_LENGTH) {
        $errors[] = 'Password must be at least ' . PASSWORD_MIN_LENGTH . ' characters long';
    }
    
    if (REQUIRE_STRONG_PASSWORD) {
        if (!preg_match('/[A-Z]/', $password)) {
            $errors[] = 'Password must contain at least one uppercase letter';
        }
        if (!preg_match('/[a-z]/', $password)) {
            $errors[] = 'Password must contain at least one lowercase letter';
        }
        if (!preg_match('/[0-9]/', $password)) {
            $errors[] = 'Password must contain at least one number';
        }
        if (!preg_match('/[!@#$%^&*(),.?":{}|<>]/', $password)) {
            $errors[] = 'Password must contain at least one special character';
        }
    }
    
    return [
        'valid' => empty($errors),
        'errors' => $errors
    ];
}

/**
 * Generate secure password reset token
 */
function generate_password_reset_token(string $email, string $user_type = 'user'): string {
    $token = bin2hex(random_bytes(32));
    $expires = date('Y-m-d H:i:s', strtotime('+1 hour'));
    
    try {
        // Invalidate old tokens
        $stmt = db()->prepare("UPDATE password_resets SET used_at = NOW() 
            WHERE email = ? AND user_type = ? AND used_at IS NULL");
        $stmt->execute([$email, $user_type]);
        
        // Create new token
        $stmt = db()->prepare("INSERT INTO password_resets 
            (email, token, user_type, ip_address, expires_at) 
            VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([
            $email,
            password_hash($token, PASSWORD_DEFAULT),
            $user_type,
            $_SERVER['REMOTE_ADDR'] ?? null,
            $expires
        ]);
        
        return $token;
    } catch (PDOException $e) {
        error_log("Failed to generate reset token: " . $e->getMessage());
        return '';
    }
}

/**
 * Verify password reset token
 */
function verify_password_reset_token(string $token, string $email, string $user_type = 'user'): bool {
    try {
        $stmt = db()->prepare("SELECT * FROM password_resets 
            WHERE email = ? AND user_type = ? AND used_at IS NULL 
            AND expires_at > NOW() 
            ORDER BY created_at DESC LIMIT 1");
        $stmt->execute([$email, $user_type]);
        $reset = $stmt->fetch();
        
        if ($reset && password_verify($token, $reset['token'])) {
            // Mark as used
            $stmt = db()->prepare("UPDATE password_resets SET used_at = NOW() WHERE id = ?");
            $stmt->execute([$reset['id']]);
            return true;
        }
        return false;
    } catch (PDOException $e) {
        return false;
    }
}

/**
 * Send email via queue
 */
function queue_email(string $to_email, string $to_name, string $subject, string $body_html, string $body_text = '', string $template_key = ''): bool {
    try {
        $stmt = db()->prepare("INSERT INTO email_queue 
            (to_email, to_name, subject, body_html, body_text, template_key, status) 
            VALUES (?, ?, ?, ?, ?, ?, 'pending')");
        $stmt->execute([$to_email, $to_name, $subject, $body_html, $body_text, $template_key]);
        return true;
    } catch (PDOException $e) {
        error_log("Failed to queue email: " . $e->getMessage());
        return false;
    }
}

/**
 * Get system setting
 */
function get_system_setting(string $key, $default = '') {
    try {
        $stmt = db()->prepare("SELECT setting_value, data_type FROM system_settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        $result = $stmt->fetch();
        
        if (!$result) {
            return $default;
        }
        
        $value = $result['setting_value'];
        $type = $result['data_type'];
        
        switch ($type) {
            case 'boolean':
                return $value === '1' || $value === 'true';
            case 'integer':
                return (int) $value;
            case 'json':
            case 'array':
                return json_decode($value, true) ?? $default;
            default:
                return $value;
        }
    } catch (PDOException $e) {
        return $default;
    }
}

/**
 * Update system setting
 */
function update_system_setting(string $key, $value, string $data_type = 'string'): bool {
    try {
        if (is_array($value) || is_object($value)) {
            $value = json_encode($value);
        } elseif (is_bool($value)) {
            $value = $value ? '1' : '0';
        }
        
        $stmt = db()->prepare("INSERT INTO system_settings 
            (setting_key, setting_value, data_type, updated_at) 
            VALUES (?, ?, ?, NOW()) 
            ON DUPLICATE KEY UPDATE 
            setting_value = VALUES(setting_value), 
            data_type = VALUES(data_type), 
            updated_at = NOW()");
        $stmt->execute([$key, $value, $data_type]);
        return true;
    } catch (PDOException $e) {
        error_log("Failed to update system setting: " . $e->getMessage());
        return false;
    }
}

/**
 * Check if user has permission
 */
function has_permission(string $permission, int $user_id = null, string $user_type = 'admin'): bool {
    if ($user_type === 'admin' && isset($_SESSION['admin_role']) && $_SESSION['admin_role'] === 'super_admin') {
        return true;
    }
    
    // Add more granular permission checks as needed
    $permissions = [
        'super_admin' => ['*'],
        'admin' => ['view_dashboard', 'manage_users', 'manage_subscriptions', 'view_reports']
    ];
    
    $role = $_SESSION['admin_role'] ?? 'admin';
    return in_array($permission, $permissions[$role] ?? []);
}

/**
 * Generate referral code
 */
function generate_referral_code(int $user_id): string {
    $code = 'HOST' . strtoupper(substr(md5($user_id . time()), 0, 6));
    
    try {
        // Check if code exists
        $stmt = db()->prepare("SELECT id FROM referrals WHERE referral_code = ?");
        $stmt->execute([$code]);
        if ($stmt->fetch()) {
            return generate_referral_code($user_id + 1);
        }
        
        // Save referral code
        $stmt = db()->prepare("INSERT INTO referrals 
            (referrer_id, referral_code, expires_at) 
            VALUES (?, ?, DATE_ADD(NOW(), INTERVAL 1 YEAR))");
        $stmt->execute([$user_id, $code]);
        
        return $code;
    } catch (PDOException $e) {
        return '';
    }
}

/**
 * Add user notification
 */
function add_notification(int $user_id, string $type, string $title, string $message, string $action_url = ''): bool {
    try {
        $stmt = db()->prepare("INSERT INTO user_notifications 
            (user_id, type, title, message, action_url) 
            VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$user_id, $type, $title, $message, $action_url]);
        return true;
    } catch (PDOException $e) {
        error_log("Failed to add notification: " . $e->getMessage());
        return false;
    }
}

/**
 * Get user notifications
 */
function get_user_notifications(int $user_id, bool $unread_only = false, int $limit = 10): array {
    try {
        $sql = "SELECT * FROM user_notifications WHERE user_id = ?";
        if ($unread_only) {
            $sql .= " AND is_read = 0";
        }
        $sql .= " ORDER BY created_at DESC LIMIT ?";
        
        $stmt = db()->prepare($sql);
        $stmt->execute([$user_id, $limit]);
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        return [];
    }
}

/**
 * Mark notifications as read
 */
function mark_notifications_read(int $user_id, array $notification_ids = []): bool {
    try {
        if (empty($notification_ids)) {
            $stmt = db()->prepare("UPDATE user_notifications SET is_read = 1, read_at = NOW() WHERE user_id = ? AND is_read = 0");
            $stmt->execute([$user_id]);
        } else {
            $placeholders = implode(',', array_fill(0, count($notification_ids), '?'));
            $stmt = db()->prepare("UPDATE user_notifications SET is_read = 1, read_at = NOW() WHERE id IN ($placeholders) AND user_id = ?");
            $params = array_merge($notification_ids, [$user_id]);
            $stmt->execute($params);
        }
        return true;
    } catch (PDOException $e) {
        return false;
    }
}

/**
 * Export data to CSV
 */
function export_to_csv(array $data, array $headers, string $filename): void {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = fopen('php://output', 'w');
    
    // Add UTF-8 BOM for Excel compatibility
    fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));
    
    // Write headers
    fputcsv($output, $headers);
    
    // Write data
    foreach ($data as $row) {
        fputcsv($output, $row);
    }
    
    fclose($output);
    exit();
}

/**
 * Validate file upload
 */
function validate_file_upload(array $file, string $type = 'image'): array {
    $result = ['valid' => false, 'error' => ''];
    
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $result['error'] = 'Upload failed: ' . $file['error'];
        return $result;
    }
    
    $max_size = MAX_FILE_UPLOAD_SIZE_MB * 1024 * 1024;
    if ($file['size'] > $max_size) {
        $result['error'] = 'File size exceeds ' . MAX_FILE_UPLOAD_SIZE_MB . 'MB limit';
        return $result;
    }
    
    if ($type === 'image') {
        $allowed_ext = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
    } elseif ($type === 'document') {
        $allowed_ext = ['pdf', 'doc', 'docx', 'xls', 'xlsx'];
    } else {
        $allowed_ext = [];
    }
    
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, $allowed_ext)) {
        $result['error'] = 'Invalid file type. Allowed: ' . implode(', ', $allowed_ext);
        return $result;
    }
    
    $result['valid'] = true;
    return $result;
}

/**
 * Get unread notification count
 */
function get_unread_notification_count(int $user_id): int {
    try {
        $stmt = db()->prepare("SELECT COUNT(*) as count FROM user_notifications WHERE user_id = ? AND is_read = 0");
        $stmt->execute([$user_id]);
        return (int) $stmt->fetch()['count'];
    } catch (PDOException $e) {
        return 0;
    }
}
