<?php
/**
 * Hostxolo - Contact Us Page
 */
require_once 'config.php';

$success = '';
$errors = [];

// Process contact form
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token.';
    } else {
        $name = sanitize($_POST['name'] ?? '');
        $email = sanitize($_POST['email'] ?? '');
        $phone = sanitize($_POST['phone'] ?? '');
        $subject = sanitize($_POST['subject'] ?? '');
        $message = sanitize($_POST['message'] ?? '');
        
        // Validation
        if (empty($name) || strlen($name) < 2) {
            $errors[] = 'Please enter your name.';
        }
        
        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Please enter a valid email address.';
        }
        
        if (empty($subject)) {
            $errors[] = 'Please enter a subject.';
        }
        
        if (empty($message) || strlen($message) < 10) {
            $errors[] = 'Please enter a message (minimum 10 characters).';
        }
        
        // Save to database
        if (empty($errors)) {
            try {
                $stmt = db()->prepare("INSERT INTO contact_inquiries (name, email, phone, subject, message, ip_address, created_at) 
                                      VALUES (?, ?, ?, ?, ?, ?, NOW())");
                $stmt->execute([$name, $email, $phone, $subject, $message, $_SERVER['REMOTE_ADDR'] ?? null]);
                
                // Send email notification to admin
                if (function_exists('send_contact_notification')) {
                    $email_sent = send_contact_notification($name, $email, $phone, $subject, $message);
                    if ($email_sent) {
                        $success = 'Thank you for your message! We will get back to you within 24 hours.';
                    } else {
                        $success = 'Thank you for your message! We will get back to you within 24 hours. (Email notification failed but message was saved)';
                    }
                } else {
                    $success = 'Thank you for your message! We will get back to you within 24 hours.';
                }
                
                // Clear form
                $_POST = [];
                
            } catch (PDOException $e) {
                $errors[] = 'Failed to send message. Please try again.';
                error_log("Contact form error: " . $e->getMessage());
            }
        }
    }
}

$csrf_token = generate_csrf_token();

include 'includes/header.php';
?>

<!-- Page Header -->
<section class="page-header">
    <div class="container">
        <h1>Contact Us</h1>
        <p>We would love to hear from you</p>
    </div>
</section>

<!-- Contact Section -->
<section class="py-5">
    <div class="container">
        <div class="row g-5">
            <!-- Contact Info -->
            <div class="col-lg-4">
                <div class="card h-100 border-0 shadow-sm">
                    <div class="card-body p-4">
                        <h4 class="fw-bold mb-4">Get in Touch</h4>
                        
                        <div class="mb-4">
                            <div class="d-flex">
                                <div class="flex-shrink-0">
                                    <div class="bg-primary bg-opacity-10 p-3 rounded" style="min-width: 60px; min-height: 60px; display: flex; align-items: center; justify-content: center;">
                                        <i class="bi bi-geo-alt text-primary fs-5" style="font-size: 1.5rem !important;"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1 ms-3">
                                    <h6 class="fw-bold mb-1">Address</h6>
                                    <p class="text-muted mb-0"><?php echo get_system_setting('contact_address', 'India'); ?></p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-4">
                            <div class="d-flex">
                                <div class="flex-shrink-0">
                                    <div class="bg-primary bg-opacity-10 p-3 rounded" style="min-width: 60px; min-height: 60px; display: flex; align-items: center; justify-content: center;">
                                        <i class="bi bi-envelope text-primary fs-5" style="font-size: 1.5rem !important;"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1 ms-3">
                                    <h6 class="fw-bold mb-1">Email</h6>
                                    <p class="text-muted mb-0"><?php echo get_system_setting('contact_email', 'support@hostxolo.com'); ?></p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-4">
                            <div class="d-flex">
                                <div class="flex-shrink-0">
                                    <div class="bg-primary bg-opacity-10 p-3 rounded" style="min-width: 60px; min-height: 60px; display: flex; align-items: center; justify-content: center;">
                                        <i class="bi bi-phone text-primary fs-5" style="font-size: 1.5rem !important;"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1 ms-3">
                                    <h6 class="fw-bold mb-1">Phone</h6>
                                    <p class="text-muted mb-0"><?php echo get_system_setting('contact_phone', '+91-6388758325'); ?></p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-4">
                            <div class="d-flex">
                                <div class="flex-shrink-0">
                                    <div class="bg-success bg-opacity-10 p-3 rounded">
                                        <i class="bi bi-whatsapp text-success fs-5"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1 ms-3">
                                    <h6 class="fw-bold mb-1">WhatsApp</h6>
                                    <p class="text-muted mb-0"><?php echo get_system_setting('social_whatsapp', '+91-6388758325'); ?></p>
                                </div>
                            </div>
                        </div>
                        
                        <hr>
                        
                        <h6 class="fw-bold mb-3">Business Hours</h6>
                        <p class="text-muted mb-1"><?php echo get_system_setting('business_hours', 'Monday - Saturday: 9:00 AM - 6:00 PM IST'); ?></p>
                        
                        <hr>
                        
                        <div class="d-flex gap-2">
                            <?php if (get_system_setting('social_facebook')): ?>
                                <a href="<?php echo get_system_setting('social_facebook'); ?>" class="btn btn-outline-primary" target="_blank"><i class="bi bi-facebook"></i></a>
                            <?php endif; ?>
                            <?php if (get_system_setting('social_twitter')): ?>
                                <a href="<?php echo get_system_setting('social_twitter'); ?>" class="btn btn-outline-primary" target="_blank"><i class="bi bi-twitter"></i></a>
                            <?php endif; ?>
                            <?php if (get_system_setting('social_instagram')): ?>
                                <a href="<?php echo get_system_setting('social_instagram'); ?>" class="btn btn-outline-primary" target="_blank"><i class="bi bi-instagram"></i></a>
                            <?php endif; ?>
                            <?php if (get_system_setting('social_linkedin')): ?>
                                <a href="<?php echo get_system_setting('social_linkedin'); ?>" class="btn btn-outline-primary" target="_blank"><i class="bi bi-linkedin"></i></a>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Contact Form -->
            <div class="col-lg-8">
                <div class="card border-0 shadow-sm">
                    <div class="card-body p-4 p-md-5">
                        <h4 class="fw-bold mb-4">Send us a Message</h4>
                        
                        <?php if ($success): ?>
                            <div class="alert alert-success">
                                <i class="bi bi-check-circle-fill me-2"></i><?php echo $success; ?>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (!empty($errors)): ?>
                            <div class="alert alert-danger">
                                <ul class="mb-0 ps-3">
                                    <?php foreach ($errors as $error): ?>
                                        <li><?php echo $error; ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        <?php endif; ?>
                        
                        <form method="POST" action="">
                            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="name" class="form-label">Your Name *</label>
                                    <input type="text" class="form-control" id="name" name="name" required
                                           value="<?php echo $_POST['name'] ?? ''; ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="email" class="form-label">Email Address *</label>
                                    <input type="email" class="form-control" id="email" name="email" required
                                           value="<?php echo $_POST['email'] ?? ''; ?>">
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="phone" class="form-label">Phone Number</label>
                                    <input type="tel" class="form-control" id="phone" name="phone"
                                           value="<?php echo $_POST['phone'] ?? ''; ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="subject" class="form-label">Subject *</label>
                                    <select class="form-select" id="subject" name="subject" required>
                                        <option value="">Select a subject</option>
                                        <option value="General Inquiry" <?php echo ($_POST['subject'] ?? '') === 'General Inquiry' ? 'selected' : ''; ?>>General Inquiry</option>
                                        <option value="Business Growth Plan" <?php echo ($_POST['subject'] ?? '') === 'Business Growth Plan' ? 'selected' : ''; ?>>Business Growth Plan</option>
                                        <option value="Software Demo" <?php echo ($_POST['subject'] ?? '') === 'Software Demo' ? 'selected' : ''; ?>>Software Demo</option>
                                        <option value="Support" <?php echo ($_POST['subject'] ?? '') === 'Support' ? 'selected' : ''; ?>>Support</option>
                                        <option value="Partnership" <?php echo ($_POST['subject'] ?? '') === 'Partnership' ? 'selected' : ''; ?>>Partnership</option>
                                        <option value="Other" <?php echo ($_POST['subject'] ?? '') === 'Other' ? 'selected' : ''; ?>>Other</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="mb-4">
                                <label for="message" class="form-label">Message *</label>
                                <textarea class="form-control" id="message" name="message" rows="5" required
                                          placeholder="Tell us how we can help you..."><?php echo $_POST['message'] ?? ''; ?></textarea>
                            </div>
                            
                            <div class="d-grid">
                                <button type="submit" class="btn btn-primary btn-lg">
                                    <i class="bi bi-send me-2"></i>Send Message
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Map Section (Placeholder) -->
<section class="py-5 bg-light">
    <div class="container">
        <div class="card border-0 shadow-sm">
            <div class="card-body p-0">
                <div class="ratio ratio-21x9">
                    <iframe src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3500000!2d78.9629!3d20.5937!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x0%3A0x0!2zMjDCsDM1JzM3LjMiTiA3OMKwNTcnNDYuNCJF!5e0!3m2!1sen!2sin!4v1234567890" 
                            width="100%" height="100%" style="border:0;" allowfullscreen="" loading="lazy">
                    </iframe>
                </div>
            </div>
        </div>
    </div>
</section>

<?php include 'includes/footer.php'; ?>
