<?php
/**
 * Hostxolo - Email Sending Function
 * SMTP email functionality using PHPMailer
 */

// Include PHPMailer autoloader
require_once __DIR__ . '/../vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\SMTP;

/**
 * Send email using SMTP
 */
function send_email($to_email, $to_name, $subject, $body_html, $body_text = '', $attachments = []) {
    try {
        $mail = new PHPMailer(true);
        
        // Server settings
        $mail->isSMTP();
        $mail->Host       = SMTP_HOST;
        $mail->SMTPAuth   = true;
        $mail->Username   = SMTP_USERNAME;
        $mail->Password   = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_ENCRYPTION;
        $mail->Port       = SMTP_PORT;
        
        // Debug settings (remove in production)
        $mail->SMTPDebug = SMTP::DEBUG_SERVER; // Enable verbose debugging
        $mail->Debugoutput = function($str, $level) {
            error_log("SMTP Debug [$level]: $str");
        };
        
        // Recipients
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($to_email, $to_name);
        
        // Add reply-to if different from sender
        if (SMTP_FROM_EMAIL !== SMTP_USERNAME) {
            $mail->addReplyTo(SMTP_USERNAME, SMTP_FROM_NAME);
        }
        
        // Attachments
        foreach ($attachments as $attachment) {
            if (is_array($attachment)) {
                $mail->addAttachment($attachment['path'], $attachment['name'] ?? '');
            } else {
                $mail->addAttachment($attachment);
            }
        }
        
        // Content
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body    = $body_html;
        $mail->AltBody = $body_text ?: strip_tags($body_html);
        
        // Send email
        $mail->send();
        
        error_log("Email sent successfully to: $to_email");
        return true;
        
    } catch (Exception $e) {
        error_log("Email sending failed: " . $e->getMessage());
        error_log("Mailer Error: " . $mail->ErrorInfo);
        return false;
    }
}

/**
 * Send welcome email
 */
function send_welcome_email($user_email, $user_name) {
    $subject = 'Welcome to Hostxolo!';
    $body_html = '
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
            <div style="background: linear-gradient(135deg, #f97316, #ea580c); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0;">
                <h1 style="margin: 0; font-size: 28px;">Welcome to Hostxolo!</h1>
                <p style="margin: 10px 0; font-size: 16px;">Your hosting journey starts here</p>
            </div>
            <div style="background: #f8f9fa; padding: 30px; border-radius: 0 0 10px 10px;">
                <h2 style="color: #333; margin-top: 0;">Hello ' . htmlspecialchars($user_name) . '!</h2>
                <p style="color: #666; line-height: 1.6;">Thank you for joining Hostxolo! Your account has been created successfully.</p>
                
                <div style="background: white; padding: 20px; border-radius: 5px; margin: 20px 0;">
                    <h3 style="color: #f97316; margin-top: 0;">What\'s Next?</h3>
                    <ul style="color: #666; line-height: 1.8;">
                        <li><strong>Login to Dashboard:</strong> Access your account and manage services</li>
                        <li><strong>Browse Plans:</strong> Choose the perfect hosting plan for your needs</li>
                        <li><strong>Get Support:</strong> Our team is here to help 24/7</li>
                    </ul>
                </div>
                
                <div style="text-align: center; margin: 30px 0;">
                    <a href="' . BASE_URL . 'login.php" style="background: #f97316; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; display: inline-block; font-weight: bold;">
                        Login to Your Account
                    </a>
                </div>
                
                <hr style="border: none; border-top: 1px solid #e9ecef; margin: 30px 0;">
                
                <p style="color: #999; font-size: 14px; text-align: center;">
                    If you have any questions, contact our support team at<br>
                    <a href="mailto:' . SMTP_FROM_EMAIL . '" style="color: #f97316;">' . SMTP_FROM_EMAIL . '</a>
                </p>
            </div>
        </div>
    ';
    
    return send_email($user_email, $user_name, $subject, $body_html);
}

/**
 * Send password reset email
 */
function send_password_reset_email($user_email, $user_name, $reset_token) {
    $reset_url = BASE_URL . 'reset-password.php?token=' . urlencode($reset_token) . '&email=' . urlencode($user_email);
    
    $subject = 'Password Reset Request - Hostxolo';
    $body_html = '
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
            <div style="background: linear-gradient(135deg, #dc3545, #c82333); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0;">
                <h1 style="margin: 0; font-size: 28px;">Password Reset</h1>
                <p style="margin: 10px 0; font-size: 16px;">Request received for your Hostxolo account</p>
            </div>
            <div style="background: #f8f9fa; padding: 30px; border-radius: 0 0 10px 10px;">
                <h2 style="color: #333; margin-top: 0;">Hello ' . htmlspecialchars($user_name) . '!</h2>
                <p style="color: #666; line-height: 1.6;">We received a request to reset your password. Click the button below to reset it:</p>
                
                <div style="text-align: center; margin: 30px 0;">
                    <a href="' . $reset_url . '" style="background: #dc3545; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; display: inline-block; font-weight: bold;">
                        Reset Password
                    </a>
                </div>
                
                <div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 5px; margin: 20px 0;">
                    <p style="color: #856404; margin: 0; font-size: 14px;">
                        <strong>Important:</strong> This password reset link will expire in 1 hour for security reasons.
                    </p>
                </div>
                
                <p style="color: #666; font-size: 14px;">If you didn\'t request this password reset, please ignore this email.</p>
                
                <hr style="border: none; border-top: 1px solid #e9ecef; margin: 30px 0;">
                
                <p style="color: #999; font-size: 14px; text-align: center;">
                    If you have any issues, contact our support team at<br>
                    <a href="mailto:' . SMTP_FROM_EMAIL . '" style="color: #f97316;">' . SMTP_FROM_EMAIL . '</a>
                </p>
            </div>
        </div>
    ';
    
    return send_email($user_email, $user_name, $subject, $body_html);
}

/**
 * Send contact form notification
 */
function send_contact_notification($name, $email, $phone, $subject, $message) {
    $notification_subject = 'New Contact Form Submission - ' . htmlspecialchars($subject);
    $body_html = '
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
            <div style="background: linear-gradient(135deg, #28a745, #1e7e34); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0;">
                <h1 style="margin: 0; font-size: 28px;">New Contact Inquiry</h1>
                <p style="margin: 10px 0; font-size: 16px;">Someone has contacted you through your website</p>
            </div>
            <div style="background: #f8f9fa; padding: 30px; border-radius: 0 0 10px 10px;">
                <h2 style="color: #333; margin-top: 0;">Contact Details:</h2>
                
                <table style="width: 100%; border-collapse: collapse;">
                    <tr>
                        <td style="background: #e9ecef; padding: 10px; font-weight: bold; width: 120px;">Name:</td>
                        <td style="padding: 10px;">' . htmlspecialchars($name) . '</td>
                    </tr>
                    <tr>
                        <td style="background: #e9ecef; padding: 10px; font-weight: bold;">Email:</td>
                        <td style="padding: 10px;"><a href="mailto:' . htmlspecialchars($email) . '">' . htmlspecialchars($email) . '</a></td>
                    </tr>
                    <tr>
                        <td style="background: #e9ecef; padding: 10px; font-weight: bold;">Phone:</td>
                        <td style="padding: 10px;">' . htmlspecialchars($phone) . '</td>
                    </tr>
                    <tr>
                        <td style="background: #e9ecef; padding: 10px; font-weight: bold;">Subject:</td>
                        <td style="padding: 10px;">' . htmlspecialchars($subject) . '</td>
                    </tr>
                </table>
                
                <div style="background: white; padding: 20px; border-radius: 5px; margin: 20px 0;">
                    <h3 style="color: #333; margin-top: 0;">Message:</h3>
                    <p style="color: #666; line-height: 1.6; white-space: pre-wrap;">' . htmlspecialchars($message) . '</p>
                </div>
                
                <div style="text-align: center; margin: 30px 0;">
                    <a href="' . BASE_URL . 'admin/inquiries.php" style="background: #28a745; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; display: inline-block; font-weight: bold;">
                        View All Inquiries
                    </a>
                </div>
                
                <hr style="border: none; border-top: 1px solid #e9ecef; margin: 30px 0;">
                
                <p style="color: #999; font-size: 14px; text-align: center;">
                    This notification was sent from ' . BASE_URL . ' at ' . date('Y-m-d H:i:s') . '
                </p>
            </div>
        </div>
    ';
    
    return send_email(SMTP_FROM_EMAIL, 'Hostxolo Admin', $notification_subject, $body_html);
}
