<?php
/**
 * Hostxolo - Payment Processing Page
 * Handles payment confirmation and status display
 */
require_once 'config.php';

// Get subscription ID and transaction ID
$subscription_id = intval($_GET['subscription'] ?? 0);
$transaction_id = sanitize($_GET['txn'] ?? '');

if (!$subscription_id || empty($transaction_id)) {
    redirect(BASE_URL . 'user/subscriptions.php', 'error', 'Invalid payment request');
}

// Verify subscription belongs to logged-in user
$user_id = $_SESSION['user_id'] ?? 0;
if (!$user_id) {
    redirect(BASE_URL . 'login.php', 'error', 'Please login to view payment status');
}

// Get subscription details
$subscription = null;
try {
    $stmt = db()->prepare("
        SELECT s.*, p.name as plan_name, p.price as plan_price, p.duration_months,
               u.full_name, u.email, u.phone
        FROM user_subscriptions s
        JOIN subscription_plans p ON s.plan_id = p.id
        JOIN users u ON s.user_id = u.id
        WHERE s.id = ? AND s.user_id = ?
    ");
    $stmt->execute([$subscription_id, $user_id]);
    $subscription = $stmt->fetch();
} catch (PDOException $e) {
    error_log("Payment page error: " . $e->getMessage());
}

if (!$subscription) {
    redirect(BASE_URL . 'user/subscriptions.php', 'error', 'Subscription not found');
}

// Get transaction details
$transaction = null;
try {
    $stmt = db()->prepare("SELECT * FROM transactions WHERE transaction_id = ? AND user_id = ?");
    $stmt->execute([$transaction_id, $user_id]);
    $transaction = $stmt->fetch();
} catch (PDOException $e) {
    error_log("Transaction fetch error: " . $e->getMessage());
}

// Determine payment status
$payment_status = $transaction['status'] ?? $subscription['payment_status'] ?? 'pending';
$is_success = in_array($payment_status, ['paid', 'completed', 'success']);
$is_failed = in_array($payment_status, ['failed', 'cancelled', 'declined']);

$page_title = $is_success ? 'Payment Successful' : ($is_failed ? 'Payment Failed' : 'Payment Status');

include 'includes/header.php';
?>

<style>
.payment-status-card {
    max-width: 600px;
    margin: 0 auto;
    border-radius: 16px;
    overflow: hidden;
}
.status-icon {
    width: 80px;
    height: 80px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2.5rem;
    margin: 0 auto 20px;
}
.status-success {
    background: #d1fae5;
    color: #065f46;
}
.status-failed {
    background: #fee2e2;
    color: #991b1b;
}
.status-pending {
    background: #fef3c7;
    color: #92400e;
}
.payment-details {
    background: #f8fafc;
    border-radius: 12px;
    padding: 20px;
}
.detail-row {
    display: flex;
    justify-content: space-between;
    padding: 10px 0;
    border-bottom: 1px solid #e5e7eb;
}
.detail-row:last-child {
    border-bottom: none;
}
</style>

<div class="container py-5">
    <div class="payment-status-card card border-0 shadow-lg">
        <div class="card-body p-5 text-center">
            <?php if ($is_success): ?>
                <div class="status-icon status-success">
                    <i class="bi bi-check-lg"></i>
                </div>
                <h2 class="fw-bold text-success mb-2">Payment Successful!</h2>
                <p class="text-muted mb-4">Thank you for your payment. Your subscription is now active.</p>
            <?php elseif ($is_failed): ?>
                <div class="status-icon status-failed">
                    <i class="bi bi-x-lg"></i>
                </div>
                <h2 class="fw-bold text-danger mb-2">Payment Failed</h2>
                <p class="text-muted mb-4">We couldn't process your payment. Please try again or contact support.</p>
            <?php else: ?>
                <div class="status-icon status-pending">
                    <i class="bi bi-hourglass-split"></i>
                </div>
                <h2 class="fw-bold text-warning mb-2">Payment Pending</h2>
                <p class="text-muted mb-4">Your payment is being processed. Please wait...</p>
            <?php endif; ?>

            <div class="payment-details text-start mb-4">
                <h5 class="fw-bold mb-3">Payment Details</h5>
                <div class="detail-row">
                    <span class="text-muted">Transaction ID</span>
                    <span class="fw-medium"><?php echo $transaction_id; ?></span>
                </div>
                <div class="detail-row">
                    <span class="text-muted">Plan</span>
                    <span class="fw-medium"><?php echo $subscription['plan_name']; ?></span>
                </div>
                <div class="detail-row">
                    <span class="text-muted">Duration</span>
                    <span class="fw-medium"><?php echo $subscription['duration_months']; ?> months</span>
                </div>
                <div class="detail-row">
                    <span class="text-muted">Amount</span>
                    <span class="fw-bold text-primary"><?php echo format_currency($subscription['amount_paid'] ?? $subscription['plan_price']); ?></span>
                </div>
                <div class="detail-row">
                    <span class="text-muted">Status</span>
                    <span class="badge bg-<?php echo $is_success ? 'success' : ($is_failed ? 'danger' : 'warning'); ?>">
                        <?php echo ucfirst($payment_status); ?>
                    </span>
                </div>
                <div class="detail-row">
                    <span class="text-muted">Date</span>
                    <span class="fw-medium"><?php echo format_date($subscription['created_at']); ?></span>
                </div>
            </div>

            <div class="d-grid gap-2 d-md-flex justify-content-md-center">
                <a href="<?php echo BASE_URL; ?>user/dashboard.php" class="btn btn-primary btn-lg">
                    <i class="bi bi-speedometer2 me-2"></i>Go to Dashboard
                </a>
                <a href="<?php echo BASE_URL; ?>user/subscriptions.php" class="btn btn-outline-primary btn-lg">
                    <i class="bi bi-card-list me-2"></i>View Subscriptions
                </a>
                <?php if ($is_success): ?>
                    <a href="<?php echo BASE_URL; ?>user/invoice.php?id=<?php echo $subscription_id; ?>" class="btn btn-outline-success btn-lg">
                        <i class="bi bi-receipt me-2"></i>Download Invoice
                    </a>
                <?php endif; ?>
            </div>

            <?php if ($is_failed): ?>
                <div class="mt-4">
                    <a href="<?php echo BASE_URL; ?>subscribe.php?plan=<?php echo $subscription['plan_id']; ?>" class="btn btn-warning btn-lg">
                        <i class="bi bi-arrow-clockwise me-2"></i>Try Again
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Support Section -->
    <div class="text-center mt-4">
        <p class="text-muted">
            <i class="bi bi-headset me-2"></i>
            Need help? <a href="<?php echo BASE_URL; ?>contact.php">Contact our support team</a>
        </p>
    </div>
</div>

<?php include 'includes/footer.php'; ?>
