<?php
/**
 * User - Enhanced My Subscriptions
 */
require_once __DIR__ . '/../config.php';
require_login();

$page_title = 'My Subscriptions';
$user_id = $_SESSION['user_id'];

$subscriptions = [];
try {
    $stmt = db()->prepare("SELECT us.*, sp.name as plan_name, sp.features, sp.description
                          FROM user_subscriptions us
                          JOIN subscription_plans sp ON us.plan_id = sp.id
                          WHERE us.user_id = ?
                          ORDER BY us.created_at DESC");
    $stmt->execute([$user_id]);
    $subscriptions = $stmt->fetchAll();
} catch (PDOException $e) {}

// Calculate stats
$active_count = count(array_filter($subscriptions, fn($s) => $s['status'] === 'active'));
$expired_count = count(array_filter($subscriptions, fn($s) => $s['status'] === 'expired'));
$total_spent = array_sum(array_column($subscriptions, 'amount'));

include __DIR__ . '/includes/header.php';
?>

<style>
.subscription-card {
    border: none;
    border-radius: 20px;
    transition: all 0.3s ease;
    overflow: hidden;
}
.subscription-card:hover {
    transform: translateY(-8px);
    box-shadow: 0 20px 40px rgba(0,0,0,0.12) !important;
}
.subscription-header {
    background: linear-gradient(135deg, #f97316 0%, #ea580c 100%);
    padding: 1.5rem;
    position: relative;
    overflow: hidden;
}
.subscription-header::after {
    content: '';
    position: absolute;
    top: -50%;
    right: -30%;
    width: 200px;
    height: 200px;
    background: radial-gradient(circle, rgba(255,255,255,0.15) 0%, transparent 70%);
    border-radius: 50%;
}
.subscription-header.active {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
}
.subscription-header.expired {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
}
.subscription-header.pending {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
}
.feature-item {
    display: flex;
    align-items: center;
    padding: 0.5rem 0;
    border-bottom: 1px dashed #e5e7eb;
}
.feature-item:last-child {
    border-bottom: none;
}
.status-badge {
    position: absolute;
    top: 1rem;
    right: 1rem;
    z-index: 10;
}
.plan-icon {
    width: 60px;
    height: 60px;
    background: rgba(255,255,255,0.2);
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.75rem;
}
.empty-state {
    text-align: center;
    padding: 4rem 2rem;
}
.empty-state-icon {
    width: 120px;
    height: 120px;
    background: linear-gradient(135deg, #eff6ff 0%, #dbeafe 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 1.5rem;
}
.stats-row {
    background: #f8fafc;
    border-radius: 16px;
    padding: 1.5rem;
    margin-bottom: 2rem;
}
</style>

<!-- Page Header -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h2 class="fw-bold mb-1">My Subscriptions</h2>
        <p class="text-muted mb-0">Manage your active plans and billing</p>
    </div>
    <a href="<?php echo BASE_URL; ?>plans.php" class="btn btn-primary btn-lg">
        <i class="bi bi-plus-circle me-2"></i>Upgrade Plan
    </a>
</div>

<?php if (!empty($subscriptions)): ?>
<!-- Stats Overview -->
<div class="stats-row">
    <div class="row g-4">
        <div class="col-md-4">
            <div class="d-flex align-items-center">
                <div class="flex-shrink-0">
                    <div class="bg-success bg-opacity-10 text-success rounded-circle d-flex align-items-center justify-content-center" style="width: 56px; height: 56px;">
                        <i class="bi bi-check-circle fs-4"></i>
                    </div>
                </div>
                <div class="flex-grow-1 ms-3">
                    <p class="text-muted mb-0 small">Active Plans</p>
                    <h4 class="fw-bold mb-0"><?php echo $active_count; ?></h4>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="d-flex align-items-center">
                <div class="flex-shrink-0">
                    <div class="bg-danger bg-opacity-10 text-danger rounded-circle d-flex align-items-center justify-content-center" style="width: 56px; height: 56px;">
                        <i class="bi bi-x-circle fs-4"></i>
                    </div>
                </div>
                <div class="flex-grow-1 ms-3">
                    <p class="text-muted mb-0 small">Expired Plans</p>
                    <h4 class="fw-bold mb-0"><?php echo $expired_count; ?></h4>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="d-flex align-items-center">
                <div class="flex-shrink-0">
                    <div class="bg-primary bg-opacity-10 text-primary rounded-circle d-flex align-items-center justify-content-center" style="width: 56px; height: 56px;">
                        <i class="bi bi-wallet2 fs-4"></i>
                    </div>
                </div>
                <div class="flex-grow-1 ms-3">
                    <p class="text-muted mb-0 small">Total Investment</p>
                    <h4 class="fw-bold mb-0"><?php echo format_currency($total_spent); ?></h4>
                </div>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<?php if (empty($subscriptions)): ?>
<!-- Empty State -->
<div class="card border-0 shadow-sm">
    <div class="empty-state">
        <div class="empty-state-icon">
            <i class="bi bi-credit-card text-primary" style="font-size: 3rem;"></i>
        </div>
        <h3 class="fw-bold mb-2">No Active Subscriptions</h3>
        <p class="text-muted mb-4">You don't have any subscriptions yet. Explore our plans to get started with powerful business tools.</p>
        <a href="<?php echo BASE_URL; ?>plans.php" class="btn btn-primary btn-lg">
            <i class="bi bi-rocket me-2"></i>View Plans
        </a>
    </div>
</div>
<?php else: ?>
<!-- Subscriptions Grid -->
<div class="row g-4">
    <?php foreach ($subscriptions as $sub): 
        $status_class = $sub['status'];
        $features = json_decode($sub['features'], true) ?? [];
        $is_active = $sub['status'] === 'active';
        
        // Calculate days remaining
        $days_remaining = 0;
        if ($is_active) {
            $end = new DateTime($sub['end_date']);
            $now = new DateTime();
            $days_remaining = max(0, $now->diff($end)->days);
        }
    ?>
    <div class="col-md-6 col-xl-4">
        <div class="card subscription-card shadow-sm h-100">
            <!-- Card Header -->
            <div class="subscription-header <?php echo $status_class; ?> text-white">
                <span class="status-badge badge bg-white text-dark">
                    <?php echo ucfirst($sub['status']); ?>
                </span>
                <div class="d-flex align-items-center position-relative" style="z-index: 1;">
                    <div class="plan-icon">
                        <i class="bi bi-star-fill text-white"></i>
                    </div>
                    <div class="ms-3">
                        <h5 class="fw-bold mb-0"><?php echo $sub['plan_name']; ?></h5>
                        <small class="opacity-90"><?php echo ucfirst($sub['subscription_type']); ?>ly Plan</small>
                    </div>
                </div>
            </div>
            
            <!-- Card Body -->
            <div class="card-body p-4">
                <!-- Price -->
                <div class="text-center mb-4">
                    <span class="display-6 fw-bold text-primary"><?php echo format_currency($sub['amount']); ?></span>
                    <span class="text-muted">/<?php echo $sub['subscription_type']; ?></span>
                </div>
                
                <?php if ($is_active && $days_remaining > 0): ?>
                <!-- Days Remaining Progress -->
                <div class="mb-4">
                    <div class="d-flex justify-content-between mb-2">
                        <small class="text-muted">Time Remaining</small>
                        <small class="fw-medium"><?php echo $days_remaining; ?> days</small>
                    </div>
                    <div class="progress" style="height: 8px;">
                        <div class="progress-bar bg-success" role="progressbar" style="width: <?php echo min(100, ($days_remaining / 30) * 100); ?>%"></div>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Plan Details -->
                <div class="mb-4">
                    <div class="d-flex justify-content-between py-2 border-bottom">
                        <span class="text-muted"><i class="bi bi-calendar me-2"></i>Start Date</span>
                        <span class="fw-medium"><?php echo format_date($sub['start_date']); ?></span>
                    </div>
                    <div class="d-flex justify-content-between py-2 border-bottom">
                        <span class="text-muted"><i class="bi bi-calendar-check me-2"></i>End Date</span>
                        <span class="fw-medium"><?php echo format_date($sub['end_date']); ?></span>
                    </div>
                    <div class="d-flex justify-content-between py-2">
                        <span class="text-muted"><i class="bi bi-cash-stack me-2"></i>Payment</span>
                        <span class="badge bg-<?php echo $sub['payment_status'] === 'paid' ? 'success' : 'warning'; ?>">
                            <?php echo ucfirst($sub['payment_status']); ?>
                        </span>
                    </div>
                </div>
                
                <!-- Features -->
                <?php if (!empty($features)): ?>
                <div class="mb-4">
                    <h6 class="fw-bold mb-3"><i class="bi bi-check2-all text-success me-2"></i>Plan Features</h6>
                    <div class="features-list">
                        <?php foreach (array_slice($features, 0, 4) as $feature): ?>
                        <div class="feature-item">
                            <i class="bi bi-check-circle-fill text-success me-2"></i>
                            <small><?php echo $feature; ?></small>
                        </div>
                        <?php endforeach; ?>
                        <?php if (count($features) > 4): ?>
                        <div class="feature-item text-muted">
                            <i class="bi bi-plus-circle me-2"></i>
                            <small>+<?php echo count($features) - 4; ?> more features</small>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Actions -->
                <div class="d-grid gap-2">
                    <a href="invoice.php?id=<?php echo $sub['id']; ?>" class="btn btn-outline-primary">
                        <i class="bi bi-download me-2"></i>Download Invoice
                    </a>
                    <?php if (!$is_active): ?>
                    <a href="<?php echo BASE_URL; ?>plans.php" class="btn btn-primary">
                        <i class="bi bi-arrow-repeat me-2"></i>Renew Plan
                    </a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    <?php endforeach; ?>
</div>

<!-- Need Help Section -->
<div class="card border-0 shadow-sm mt-4 bg-light">
    <div class="card-body p-4">
        <div class="row align-items-center">
            <div class="col-md-8">
                <h5 class="fw-bold mb-2"><i class="bi bi-question-circle text-primary me-2"></i>Need Help with Your Subscription?</h5>
                <p class="text-muted mb-0">Our support team is ready to assist you with any billing or subscription questions.</p>
            </div>
            <div class="col-md-4 text-md-end mt-3 mt-md-0">
                <a href="create-ticket.php" class="btn btn-primary">
                    <i class="bi bi-headset me-2"></i>Contact Support
                </a>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<?php include __DIR__ . '/includes/footer.php'; ?>

